/*
amethyst: media computer

Raspberry Pi 4B (4 GB RAM)

- use mainline kernel, the raspberrypi/linux upstream variant nixpkgs doesn't boot for me
- seems to have some other problems as well: https://github.com/NixOS/nixpkgs/issues/123725
- most relevant drivers have landed in the mainline kernel by now, so it should be fine
- state of mainline support for the rpi4: https://github.com/lategoodbye/rpi-zero/issues/43
*/

{ pkgs, ... }:

{
  imports = [ ../nixos ];

  # taken from https://github.com/NixOS/nixos-hardware/issues/631#issuecomment-1584100732
  boot.initrd.availableKernelModules = [
    "usbhid"
    "usb_storage"
    "vc4"
    "pcie_brcmstb" # required for the pcie bus to work
    "reset-raspberrypi" # required for vl805 firmware to load
  ];

  /*
  Fix HDMI broken hotplugging.

  - without this, HDMI hotplugging fails for my TV
  - the TV is correctly connected after boot or restart of Kodi/Cage
  - but after plugging it out and in again or turning the TV off and on again, it's not reconnected
  - this kernel parameter forces the HDMI output I use for my TV (HDMI-A-1) to always be enabled
  - that's the first one beside the power supply USB-C port
  - with the correct resolution and framerate
  - that sidesteps the need correct hotplug detection
  - the secret is the last 'D': it forces the output to always be enabled (in digital mode)

  See:
  - https://pip.raspberrypi.com/categories/685-whitepapers-app-notes/documents/RP-004341-WP/Troubleshooting-KMS-HDMI-output.pdf
    (under Mitigations)
  - https://www.kernel.org/doc/Documentation/fb/modedb.txt
  */
  boot.kernelParams = [ "video=HDMI-A-1:1920x1080@60D" ];

  hardware.enableRedistributableFirmware = true;
  powerManagement.cpuFreqGovernor = "ondemand";
  hardware.graphics.enable = true;

  /*
  Set up devicetrees to load during boot.

  - while on x86 hardware is mostly auto-discovered through ACPI, this isn't the case for other
    platforms
  - some hardware can't be discovered automatically by the kernel even if the driver is available
  - the info for those devices has to be supplied as devicetree to the kernel
  - normally that's done by the boot loader
  - e.g. on normal rpi setups the devicetree data passed to the kernel is configured in config.txt
  - general resources about devicetree:
    - https://en.wikipedia.org/wiki/Devicetree
    - https://www.raspberrypi.com/documentation/computers/configuration.html#part1
  - device trees are built from various types of files
    - .dtb: device tree binary, compiled from .dts device tree source
    - .dtbo: device tree binary overlays, also compiled from .dts
  - overlays can be applied on top of the main device tree, see below
  */
  hardware.deviceTree = let
    /*
    Pin the raspberrypifw package to an old version.

    - later versions bricked the devicetree so that HDMI wouldn't work anymore
    - due to some changes around HDMI and audio for the rpi5?
    - see: https://github.com/pftf/RPi4/issues/252
    - the uefi firmware rolled back to a version of the raspberrypi/firmware repo before those
      changes: https://github.com/pftf/RPi4/commit/6ba22a07bf19422c199dd801d3442319c04f5090
    - they pinned it to b49983637106e5fb33e2ae60d8c15a53187541e4, so I'm doing the same on the
      system level
    */
    raspberrypifw = pkgs.raspberrypifw.overrideAttrs {
      version = "pinned-2023.05.12";
      src = pkgs.fetchFromGitHub {
        owner = "raspberrypi";
        repo = "firmware";
        rev = "b49983637106e5fb33e2ae60d8c15a53187541e4";
        hash = "sha256-Ia+pUTl5MlFoYT4TrdAry0DsoBrs13rfTYx2vaxoKMw=";
      };
    };
  in {
    enable = true;

    # use the devicetree files from the official raspberrypi linux tree
    dtbSource = pkgs.unstable.device-tree_rpi.override { inherit raspberrypifw; };
    filter = "bcm2711-rpi-4-b.dtb";  # only apply overlays on the one devicetree I actually need
    name = "broadcom/bcm2711-rpi-4-b.dtb";  # use and load the correct rpi4b devicetree

    /*
    Devicetree overlays applied to the main devicetree.

    - these are applied during build, i.e. these are *not* passed separately to the bootloader
    - they're compiled in beforehand, the resulting single dtb file is then given to the bootloader
    - overlays are applied in the order they're in the list
    - the upstream overlays by raspberrypi are in the raspberrypifw package, see `upstreamOverlay`
    - these are documented in their repo: https://github.com/raspberrypi/linux/blob/rpi-6.6.y/arch/arm/boot/dts/overlays/README

    See:
    - https://docs.zephyrproject.org/latest/build/dts/intro-syntax-structure.html
    - https://bootlin.com/blog/enabling-new-hardware-on-raspberry-pi-with-device-tree-overlays/
    - https://elinux.org/Device_Tree_Source_Undocumented (for `/delete-property/`)
    */
    overlays = let
      upstreamOverlay = name: raspberrypifw + /share/raspberrypi/boot/overlays/${name}.dtbo;
    in [

      /*
      Fixes a bug I experienced with the mainline kernel where only one CPU core would work.

      - I wrote this overlay to apply the changes from this patch I found online:
        https://github.com/AntonioND/rpi3-arm-tf-bootstrap/blob/master/0001-rpi3-Enable-PSCI-support.patch
      - there'd be "failed to come online" messages in `dmesg`
      - I found the patch here: https://github.com/OP-TEE/build/issues/360
      */
      {
        name = "custom-enable-method";
        dtsText = ''
          /dts-v1/;
          /plugin/;

          / {
            compatible = "brcm,bcm2711";

            fragment@0 {
              target = <&cpus>;
              __overlay__ {
                /delete-property/ enable-method;
              };
            };

            fragment@1 {
              target-path = "/";
              __overlay__ {
                psci {
                  compatible = "arm,psci-1.0", "arm,psci-0.2";
                  method = "smc";
                };
              };
            };

            fragment@2 {
              target = <&cpu0>;
              __overlay__ {
                enable-method = "psci";
                /delete-property/ cpu-release-addr;
              };
            };

            fragment@3 {
              target = <&cpu1>;
              __overlay__ {
                enable-method = "psci";
                /delete-property/ cpu-release-addr;
              };
            };

            fragment@4 {
              target = <&cpu2>;
              __overlay__ {
                enable-method = "psci";
                /delete-property/ cpu-release-addr;
              };
            };

            fragment@5 {
              target = <&cpu3>;
              __overlay__ {
                enable-method = "psci";
                /delete-property/ cpu-release-addr;
              };
            };

          };
        '';
      }

      /*
      This overlay comprises the vc4-kms-v3d-pi4 and dwc2 overlays for graphics and USB.

      - note that nixos-hardware uses the vc4-fkms-v3d-pi4 overlay instead
      - fkms (fake/firmware kernel mode setting) uses a feature in the firmware blob, proper kms
        implements mode setting itself
      - https://forums.raspberrypi.com/viewtopic.php?t=255478
      - the kms driver is newer, fkms is deprecated by upstream now
      - fkms didn't work for me anyway
      - kms does, but only with the pinned devicetree repo (see above)
      */
      {
        name = "upstream-pi4";
        dtboFile = upstreamOverlay "upstream-pi4";
      }

    ];
  };

  eisfunke = {
    encryption = false;  # don't encrypt root to save resources, there's nothing critical there
    headless = true;
    minimal = true;
    domains = [
      { domain = "eisfunke.com"; subs = [ "amethyst" ]; type = "ipv6"; }
    ];
  };

  networking.hostName = "amethyst";
  networking.wireless.enable = true;
  systemd.network = {
    links = {
      "10-wired0" = {
        matchConfig.PermanentMACAddress = "dc:a6:32:18:b3:3d";
        linkConfig.Name = "wired0";
      };
      "10-wireless0" = {
        matchConfig.PermanentMACAddress = "dc:a6:32:18:b3:40";
        linkConfig.Name = "wireless0";
      };
    };
    networks = {
      "20-wired0" = {
        name = "wired0";
        DHCP = "ipv4";

        /*
        increase priority = lower route metrics for the routes for the wired0 network
        so the wired connection is preferred over the wireless one if available
        see https://unix.stackexchange.com/questions/555500/how-to-change-the-order-of-the-default-gateway-with-systemd-networkd
        */
        dhcpV4Config = {
          RouteMetric = 1020;  # default is 1024
        };
        ipv6AcceptRAConfig = {
          RouteMetric = 508;  # default is "512:1024:2048"
        };
      };
      "20-wireless0" = {
        name = "wireless0";
        DHCP = "ipv4";
        networkConfig = {
          IgnoreCarrierLoss = "3s";
        };
      };
    };
  };

  system.stateVersion = "22.11";

  home-manager.users.eisfunke = {
    imports = [ ../home ];
    eisfunke = {
      deviceColor = "#CC068E";  # pink
    };
  };
}
