/*
miranda: my laptop
Framework Laptop 16 (AMD Ryzen 7040 Series)

Notes:
- there's a kernel module for Framework laptops: https://github.com/DHowett/framework-laptop-kmod
  - this exposes some device-specific things to the OS
    - the battery charge limit (the maximum percentage the battery is charged to)
    - some LEDs, currently only the keyboard backlight
    - fan control
    - privacy switches
  - on the Framework 16 the keyboard backlight is actually controlled through the QMK firmware
  - I don't really need the other stuff, which can still be managed with framework-tool anyway
  - so I'm avoiding to install an extra 3rd party kernel module for now
- the keyboard uses QMK firmware
  - keyboard keymap / lighting can be managed with VIA (e.g. https://keyboard.frame.work)
  - with numlock off, + and enter on the numpad manage backlight
  - that's problematic with Neo2 because it forces numlock off
  - can be worked around by unmapping those keys, which I did
- in nixos-hardware there's an udev rule titled "Ethernet expansion card support"
  - it sets the `power/autosuspend` attribute
  - the card works for me, but if I run into problems, look into that again
- there's a udev rule for disabling waking up from the keyboard
  - https://wiki.archlinux.org/title/Framework_Laptop_16#Prevent_waking_up_in_backpack
  - supposed to help with the laptop waking in backpacks from screen flexing onto the keyboard
  - haven't had problems with that, but keep in mind
- I don't use the fingerprint reader, but support could be activated through `services.fprintd`
- there's a config file for lm_sensors in the Framework fork of lm_sensors:
  https://github.com/FrameworkComputer/lm-sensors/tree/framework/configs/Framework
- TODO: set power profile depending on power supply status
*/

{ pkgs, ... }:

{
  imports = [ ../nixos ];

  boot.kernelModules = [ "kvm-amd" ];

  /*
  Use the amd_pstate_epp frequency driver for best energy usage. The frequency governor only
  provides a hint to this driver: whether to bias towards power efficiency or performance.

  See https://wiki.archlinux.org/title/CPU_frequency_scaling
  */
  boot.kernelParams = [ "amd_pstate=active" ];
  powerManagement.cpuFreqGovernor = "powersave";

  hardware = {
    cpu.amd.updateMicrocode = true;
    enableRedistributableFirmware = true;

    # for backlight sensor support (not using that right now, but make it available anyway)
    sensor.iio.enable = true;

    # enable non-root access to QMK keyboard modules through qmk-udev-rules package
    keyboard.qmk.enable = true;
  };

  # see /docs/graphics.md
  hardware.graphics.enable = true;
  hardware.amdgpu.opencl.enable = true;
  environment.variables = {
    LIBVA_DRIVER_NAME = "radeonsi";
    VDPAU_DRIVER = "radeonsi";
  };

  /*
  Supposedly very important for good battery life.

  See https://knowledgebase.frame.work/en_us/optimizing-ubuntu-battery-life-Sye_48Lg3
  */
  services.power-profiles-daemon.enable = true;

  eisfunke.mobile = true;

  networking.hostName = "miranda";
  networking.wireless.enable = true;
  systemd.network = {
    links."10-wireless0" = {
      matchConfig.PermanentMACAddress = "fc:b0:de:17:f6:d9";
      linkConfig = {
        Name = "wireless0";
        MACAddressPolicy = "random";
      };
    };
    networks = {
      "20-wireless0" = {
        name = "wireless0";
        DHCP = "yes";
        networkConfig = {
          IPv6PrivacyExtensions = true;
          IgnoreCarrierLoss = "3s";
        };
      };
      /*
      Should match any USB ethernet adapter (and nothing else) and configure it for internet access.
      We don't use a link file for this because there could be different and/or multiple USB adapters connected
      */
      "30-usb" = {
        matchConfig = {
          Type = "ether";
          Path = "*usb*";
        };
        DHCP = "ipv4";

        /*
        increase priority = lower route metrics for the routes for the wired0 network
        so the wired connection is preferred over the wireless one if available
        see https://unix.stackexchange.com/questions/555500/how-to-change-the-order-of-the-default-gateway-with-systemd-networkd
        */
        dhcpV4Config = {
          RouteMetric = 1020;  # default is 1024
        };
        ipv6AcceptRAConfig = {
          RouteMetric = 508;  # default is "512:1024:2048"
        };
      };
    };
  };

  system.stateVersion = "23.11";

  home-manager.users.eisfunke = {
    imports = [ ../home ];

    /*
    Install Framework management tool.

    - currently needs `--driver portio`
    - see https://wiki.archlinux.org/title/Framework_Laptop_13#Interact_with_system
    */
    home.packages = [ pkgs.framework-tool ];

    eisfunke = {
      deviceColor = "#640060";  # violet
      displays = {
        main = {
          id = "eDP-1";
          posX = 0;
          posY = 0;
          width = 2560;
          height = 1600;
          scale = "1.333333333333333333333333";
        };
        secondary = null;
        auxiliary = [
          {
            id = "HDMI-A-2";
            posX = 1920;
            posY = 0;
          }
          {
            id = "DP-2";
            posX = 1920;
            posY = 0;
          }
          {
            id = "DP-3";
            posX = 1920;
            posY = 0;
          }
        ];
      };
    };
  };
}
