/*
obsidian: my off-site backup server, a Raspberry Pi 4 at my parent's house

TODO: install and fix up emerald
*/

{ lib, sshPublicKeys, ... }:

{
  imports = [ ../nixos ];

  # taken from https://github.com/NixOS/nixos-hardware/issues/631#issuecomment-1584100732
  boot.initrd.availableKernelModules = [
    "usbhid"
    "usb_storage"
    "vc4"
    "pcie_brcmstb" # required for the pcie bus to work
    "reset-raspberrypi" # required for vl805 firmware to load
  ];

  hardware.enableRedistributableFirmware = true;
  powerManagement.cpuFreqGovernor = "ondemand";

  # don't encrypt root to save resources, there's nothing critical there
  eisfunke = {
    encryption = false;
    headless = true;
    minimal = true;
    domains = [
      { domain = "eisfunke.com"; subs = [ "obsidian" ]; type = "ipv6"; }
    ];
  };

  fileSystems = let
    beryllium = "/dev/disk/by-uuid/04b87550-81cc-4d11-8723-402f6d753595";
    rubidium = "/dev/disk/by-uuid/7537e1c5-3322-4f3a-9c64-c3659beb5655";
  in {
    "/var/backup" = {
      device = beryllium;
      fsType = "btrfs";
      options = [ "nofail" "subvol=/backup" ];  # no compression, borg does that already
    };
    "/mnt/beryllium" = {
      device = beryllium;
      fsType = "btrfs";
      options = [ "nofail" "subvol=/" "compress=zstd" ];
    };
    "/mnt/rubidium" = {
      device = rubidium;
      fsType = "btrfs";
      options = [ "nofail" "subvol=/" "compress=zstd" ];
    };
  };

  networking.hostName = "obsidian";
  networking.wireless.enable = true;
  systemd.network = {
    links = {
      "10-wired0" = {
        matchConfig.PermanentMACAddress = "e4:5f:01:27:f2:27";
        linkConfig.Name = "wired0";
      };
      "10-wireless0" = {
        matchConfig.PermanentMACAddress = "e4:5f:01:27:f2:28";
        linkConfig.Name = "wireless0";
      };
    };
    networks = {
      "20-wired0" = {
        name = "wired0";
        DHCP = "ipv4";

        /*
        increase priority = lower route metrics for the routes for the wired0 network
        so the wired connection is preferred over the wireless one if available
        see https://unix.stackexchange.com/questions/555500/how-to-change-the-order-of-the-default-gateway-with-systemd-networkd
        */
        dhcpV4Config = {
          RouteMetric = 1020;  # default is 1024
        };
        ipv6AcceptRAConfig = {
          RouteMetric = 508;  # default is "512:1024:2048"
        };
      };
      "20-wireless0" = {
        name = "wireless0";
        DHCP = "ipv4";
        networkConfig = {
          IgnoreCarrierLoss = "3s";
        };
      };
    };
  };

  services.borgbackup.repos = let
    mkRepoConfig = host: {
      /*
      Only the backed-up host itself has access to its borg repo via its borg SSH key. That's a
      dedicated key managed through the config itself so it's easier to separate from "normal"
      usage.

      Access is given only in append-only mode. This means no changes or deletions on existing data
      can be performed. This also means prune or delete don't actually delete any data in the repo
      yet. Instead they only *tag* data as deleted.

      If you write to the repo without append-only mode in any way, pending transaction will
      automatically and permanently be committed. So, to free pruned storage, `borg compact` has to
      be run regularly in non-append-only mode. The transaction log should be checked and the repo
      data verified beforehand to ensure that an attacker hasn't silently corrupted the data.

      This can be achived by using borg with a normal user key with access instead of through a
      borg-specific key. Note that in that case, the full repo path has to be specified.

      ```
      ssh borg@obsidian.eisfunke.com cat /var/backup/melissa/transactions  # check log
      borg check --verify-data borg@obsidian.eisfunke.com:/var/backup/$REPO  # verify data
      borg compact borg@obsidian.eisfunke.com:/var/backup/$REPO  # compact repo
      ```

      For normal usage use `borgbackup-job-persist` instead, that uses the borg-specific SSH key
      with append-only mode enabled. It will also automatically load the passphrase.

      See: https://borgbackup.readthedocs.io/en/stable/usage/notes.html#append-only-mode-forbid-compaction
      */
      ${host} = {
        authorizedKeysAppendOnly = [ sshPublicKeys.borg.${host} ];
        path = "/var/backup/${host}";
      };
    };
  in lib.mkMerge (builtins.map mkRepoConfig [ "melissa" "miranda" "sapphire" "amethyst" ]);

  /*
  Overwrite the default authorized SSH keys and only allow my two main devices as hardening measure.
  Particularly, my homeserver sapphire is deliberately not included here. This is to protect the
  integrity of my backups.

  If my homeserver were to be compromised and an attacker gained full access, they couldn't delete
  my backups through the borg connection because that's set to be append-only. This wouldn't help
  though if the homeserver had full access to the backup device anyway and could just `rm -r` the
  entire repo from there.
  */
  users.users.eisfunke.openssh.authorizedKeys.keys = lib.mkForce [
    sshPublicKeys.user.melissa sshPublicKeys.user.miranda
  ];

  /*
  Add my main user keys as "normal" authorized keys to the borg user as well instead of in the
  borg repos above so they can access all repos and without append-only.

  They can *access* them anyway through the eisfunke user and sudo, but this makes it handier to
  administrate borg repos remotely from those devices.
  */
  users.users.borg.openssh.authorizedKeys.keys = [
    sshPublicKeys.user.melissa sshPublicKeys.user.miranda
  ];

  system.stateVersion = "24.05";

  home-manager.users.eisfunke = {
    imports = [ ../home ];
    eisfunke = {
      deviceColor = "#1D1D1D";  # dark gray
    };
  };
}
