/*
sapphire: my homeserver

Notes:
- the fast boot BIOS setting should be disabled, otherwise the keyboard will not be available in the
  bootloader
*/

{ config, pkgs, lib, ... }:

{
  imports = [ ../nixos ../nixos/server ];

  boot.initrd.availableKernelModules = [
    "nvme" "ahci"  # storage drivers
    "igb"  # network driver to enable remote encryption unlock
  ];
  # force-load the zfs module so the encrypted zpool can be imported over ssh during boot
  boot.initrd.kernelModules = [ "zfs" ];
  boot.kernelModules = [ "kvm-amd" ];
  powerManagement.cpuFreqGovernor = "ondemand";
  hardware.cpu.amd.updateMicrocode = true;
  hardware.enableRedistributableFirmware = true;

  /*
  sapphire still uses ZFS, so we must use a kernel that's compatible with the ZFS version in
  nixpkgs here.

  If we were to just use the latest compatible kernel, we could run into a problem where the latest
  supported kernel would become EOL, leading it to be removed from nixpkgs, and then having to
  downgrade to the latest LTS instead. This is why `boot.zfs.package.latestCompatibleLinuxPackages`
  was removed: https://discourse.nixos.org/t/zfs-latestcompatiblelinuxpackages-is-deprecated

  So we use the latest compatible LTS release here.
  - find ZFS version in nixpkgs: https://search.nixos.org/packages?type=packages&query=zfs
  - find kernels supported by ZFS in the release notes: https://github.com/openzfs/zfs/releases
  - find latest compatible LTS release: https://www.kernel.org/category/releases.html
  */
  boot.kernelPackages = lib.mkForce pkgs.linuxPackages_6_12;

  # ZFS stuff
  boot.supportedFilesystems = [ "zfs" ];
  boot.initrd.supportedFilesystems = [ "zfs" ];  # needed for zpool to be available during boot for unlocking
  boot.zfs.forceImportRoot = false;  # only enabled by default for compatibility
  services.zfs.autoScrub.enable = true;
  systemd.services.zfs-mount.enable = false;

  fileSystems = {
    "/var/media" = {
      device = "media/enc01";
      fsType = "zfs";
    };
    "/var/cloud" = {
      device = "media/enc02";
      fsType = "zfs";
    };
  };

  # include the zpool import in remote unlock script
  boot.initrd.systemd.extraBin.unlock = lib.mkForce (pkgs.writeScript "unlock" ''
    #/bin/sh
    zpool import -la
    systemctl start systemd-cryptsetup@main
  '');

  eisfunke.headless = true;
  eisfunke.domains = [
    {
      domain = "eisfunke.com";
      type = "both";
      subs = [
        "sapphire"
        "" "www" "gl" "cert"  # custom GitLab Pages
        "*.pages"  # GitLab pages
        "days" "matrix" "lists" "diagrams" "events" "gallery" "pad"
        "mock" "bytegeschichten" "git" "registry" "cloud" "flix" "shelf"
        "news" "dashboard" "literature" "proxy"
        "data" "auth" "gca4hpx" "vault" "nixcache" "documents" "notify"

        "forum" "lab"  # deprecated, kept for redirects
      ];
    }
    {
      domain = "lightfold.dev";
      type = "both";
      subs = [ "" "www" ];
    }
    {
      domain = "online-lenz.de";
      type = "both";
      subs = [ "" "www" ];
    }
    {
      domain = "systemproblem.de";
      type = "both";
      subs = [ "" "www" ];
    }
    {
      domain = "inductive.space";
      type = "both";
      subs = [ "" "www" ];
    }
    {
      domain = "a-way.world";
      type = "both";
      subs = [ "" "www" ];
      service = "strato";
    }
  ];

  networking = {
    hostName = "sapphire";
    hostId = "40c0de22";
  };
  systemd.network = {
    links = {
      "10-wired0" = {
        matchConfig.PermanentMACAddress = "70:85:c2:51:26:a1";
        linkConfig = {
          Name = "wired0";
          WakeOnLan = "magic";
        };
      };
      "10-wireless0" = {
        matchConfig.PermanentMACAddress = "34:f6:4b:ee:50:f2";
        linkConfig.Name = "wireless0";
      };
    };
    networks."20-wired0" = {
      name = "wired0";
      DHCP = "no";  # IPv4 and DNS is configured manually below
      address = [ "10.1.1.2/16" ];
      gateway = [ "10.1.1.1" ];
      # use 1.1.1.1 DNS directly (skip router) for more speed & reliability
      ipv6AcceptRAConfig.UseDNS = false;
      dns = [ "2606:4700:4700::1111#one.one.one.one" "2606:4700:4700::1001#one.one.one.one" ];
    };
  };
  services.resolved.dnsovertls = "true";

  system.stateVersion = "23.05";

  home-manager.users.eisfunke = {
    imports = [ ../home ];
    eisfunke = {
      deviceColor = "#102298";  # blue
    };
  };
}
