#!/bin/env python3
import datetime
import json
import re
from argparse import ArgumentParser
from datetime import timedelta
from io import BytesIO
from pprint import pprint

import requests
from dateutil import parser as date_parser
from pytz import UTC


class Date:
    weekday = ""
    day = ""
    month = ""
    year = ""
    time = ""

    def __init__(self):
        pass

class NextEvent:
    date = Date()
    title = ""
    description = ""
    movie_description = ""
    tags = ""
    link = ""
    image = ""

    days = ["Montag", "Dienstag", "Mittwoch", "Donnerstag", "Freitag", "Samstag", "Sonntag"]
    month = [
        "Januar",
        "Februar",
        "März",
        "April",
        "Mai",
        "Juni",
        "Juli",
        "August",
        "September",
        "Oktober",
        "November",
        "Dezember"
    ]

    def __init__(self, next_event, now):
        # passende Zeitangabe raussuchen
        event_date = date_parser.parse(next_event["date"])
        tomorrow = now + timedelta(days=1)
        if event_date.day == now.day and event_date.month == now.month and event_date.year == now.year:
            start = "Heute"
        elif event_date.day == tomorrow.day and event_date.month == tomorrow.month and event_date.year == tomorrow.year:
            start = "Morgen"
        else:
            start = self.days[event_date.weekday()]

        self.date.weekday = start
        self.date.day = event_date.day
        self.date.month = self.month[event_date.month - 1]
        self.date.year = event_date.year
        self.date.time = event_date.strftime("%H:%M")

        # Title des Events
        self.title = next_event["title"]

        # Beschreibung des Events
        self.description = next_event["description"]

        # Beschreibung des Films
        self.movie_description = next_event["movie"]["description"]

        # Baue die Tag-Reihe zusammen.
        self.tags = " | ".join([w["description"] for w in next_event["tags"]])

        # Generiere die URL zum Event, da diese nicht im JSON steht.
        link = next_event["title"].lower().replace('ö', 'o').replace('ä', 'a').replace('ü', 'u').replace('ß', 'ss')
        link = re.sub("[^[A-Za-z0-9]", '-', link)
        link = re.sub(r"-{2,}", '-', link)
        link: str = link.rstrip("-")
        link = str(next_event["id"]) + "-" + link
        self.link = "https://ufc.tu-dortmund.de/vorstellung/%s" % link

        if next_event["movie"]["youtube_id"] is not "":
            self.trailer = "https://youtu.be/%s" % next_event["movie"]["youtube_id"]

        response = requests.get("https://ufc.tu-dortmund.de/%s"
                                % next_event['movie']['image_url_medium']
                                .replace('medium', 'original'))
        self.image = BytesIO(response.content)


class ProgramInfo:
    semester = ""

    next_event = ""

    def __init__(self):
        # Hole das UFC Semester
        self.semester = json.loads(
            requests.get(url="https://ufc.tu-dortmund.de/semesters/current.json").text
        )

    def get_next_event(self, wanted_day_diff, just_print=False):
        # Finde das nächste Event, dass nach heute ist und in `day_diff` Tagen stattfindet.
        now = UTC.localize(
            datetime.datetime.now()
        )
        for i in self.semester:
            d = date_parser.parse(i["date"])
            if  d < now:
                continue

            diff = (d - now)
            if now.date() == d.date():
                day_diff = 0
            elif now.time() > d.time():
                day_diff = diff.days + 1
            elif now.time() <= d.time():
                day_diff = diff.days

            if day_diff == wanted_day_diff:
                if just_print:
                    pprint(i)
                else:
                    self.next_event = NextEvent(i, now)
                    return self.next_event

    def get_image(self):
        # return image data
        return self.next_event.image


def main():
    # Parse Arguments
    aparser: ArgumentParser = ArgumentParser(
        description="UFC Programminfo Bot"
    )
    aparser.add_argument(
        "--preview",
        dest='day_diff',
        action="store_const",
        const=7,
        default=1,
        help="Überprüfe die nächsten 7 Tage statt nur den nächsten Tag."
    )
    args = aparser.parse_args()

    p_info = ProgramInfo()

    p_info.get_next_event(args.day_diff, True)


if __name__ == "__main__":
    main()
